<?php
/**
 * Main plugin class
 */
class WooTsap {
    /**
     * Initialize the plugin
     */
    public function init() {
        // Register activation hook
        register_activation_hook(WOOTSAP_PLUGIN_FILE, array($this, 'activate'));
        
        // Enqueue admin scripts and styles
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
        
        // Add admin menu
        add_action('admin_menu', array($this, 'add_admin_menu'));
        
        // Register settings
        add_action('admin_init', array($this, 'register_settings'));
        
        // Remove individual status hooks and use unified hook
        // add_action('woocommerce_order_status_processing', array($this, 'send_processing_notification'), 10, 1);
        // add_action('woocommerce_order_status_completed', array($this, 'send_completed_notification'), 10, 1);
        
        // Universal hook for ALL status changes
        add_action('woocommerce_order_status_changed', array($this, 'order_status_changed'), 10, 3);
        
        // Add order notes
        add_action('woocommerce_checkout_before_order_review_heading', array($this, 'add_order_note_checkbox'));
        add_action('woocommerce_checkout_update_order_meta', array($this, 'save_order_note_checkbox'));

        // Add WhatsApp phone field to checkout
        add_filter('woocommerce_checkout_fields', array($this, 'add_whatsapp_phone_field'));
        add_action('woocommerce_checkout_update_order_meta', array($this, 'save_whatsapp_phone_field'));
        
        // Add AJAX handler for the test button
        add_action('wp_ajax_wootsap_test_api', array($this, 'test_api_ajax_handler'));
        
        // Add meta box to order screen
        add_action('add_meta_boxes', array($this, 'add_order_meta_box'));
        
        // Add AJAX handler for manual notification
        add_action('wp_ajax_wootsap_send_manual', array($this, 'send_manual_notification_ajax'));
    }

    /**
     * Plugin activation
     */
    public function activate() {
        // Check if WooCommerce is active
        if (!class_exists('WooCommerce')) {
            deactivate_plugins(plugin_basename(WOOTSAP_PLUGIN_FILE));
            wp_die(__('WooTsap requires WooCommerce to be installed and active.', 'wootsap'));
        }
        
        // Set default values for messages
        if (empty(get_option('wootsap_processing_message'))) {
            update_option('wootsap_processing_message', 'Hello {customer_name}, your order #{order_number} has been received and is being processed. Total: {order_total}. Thank you for shopping with us!');
        }
        
        if (empty(get_option('wootsap_completed_message'))) {
            update_option('wootsap_completed_message', 'Great news {customer_name}! Your order #{order_number} has been completed and is on its way to you. Thank you for shopping with us!');
        }
        
        if (empty(get_option('wootsap_default_country_code'))) {
            update_option('wootsap_default_country_code', '91');
        }
        
        // Register settings for all WooCommerce order statuses
        if (function_exists('wc_get_order_statuses')) {
            $order_statuses = wc_get_order_statuses();
            foreach ($order_statuses as $status_key => $status_label) {
                $status = str_replace('wc-', '', $status_key);
                
                // Set default messages for key statuses
                switch ($status) {
                    case 'cancelled':
                        if (empty(get_option('wootsap_' . $status . '_message'))) {
                            update_option('wootsap_' . $status . '_message', 'Hello {customer_name}, your order #{order_number} has been cancelled. If you have any questions, please contact us.');
                        }
                        break;
                    case 'on-hold':
                        if (empty(get_option('wootsap_' . $status . '_message'))) {
                            update_option('wootsap_' . $status . '_message', 'Hello {customer_name}, your order #{order_number} has been placed on hold. Total: {order_total}. We\'ll notify you when it\'s processing again.');
                        }
                        break;
                    case 'refunded':
                        if (empty(get_option('wootsap_' . $status . '_message'))) {
                            update_option('wootsap_' . $status . '_message', 'Hello {customer_name}, your order #{order_number} has been refunded. The amount of {order_total} will be returned to your payment method.');
                        }
                        break;
                    case 'failed':
                        if (empty(get_option('wootsap_' . $status . '_message'))) {
                            update_option('wootsap_' . $status . '_message', 'Hello {customer_name}, unfortunately your order #{order_number} has failed. Please try again or contact us for assistance.');
                        }
                        break;
                    case 'pending':
                        if (empty(get_option('wootsap_' . $status . '_message'))) {
                            update_option('wootsap_' . $status . '_message', 'Hello {customer_name}, your order #{order_number} is pending payment. Total: {order_total}. Please complete the payment to process your order.');
                        }
                        break;
                }
            }
        }
    }
    
    /**
     * Enqueue admin scripts and styles
     */
    public function enqueue_admin_assets($hook) {
        // Only enqueue on our settings page or order edit screen
        if (strpos($hook, 'wootsap') !== false || $hook == 'post.php') {
            $screen = get_current_screen();
            
            wp_enqueue_style('wootsap-admin', WOOTSAP_PLUGIN_URL . 'assets/css/admin.css', array(), WOOTSAP_VERSION);
            wp_enqueue_script('wootsap-admin', WOOTSAP_PLUGIN_URL . 'assets/js/admin.js', array('jquery'), WOOTSAP_VERSION, true);
            
            wp_localize_script('wootsap-admin', 'wootsap_params', array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('wootsap-admin'),
                'testing_text' => __('Testing connection...', 'wootsap'),
                'sending_text' => __('Sending...', 'wootsap'),
                'success_text' => __('Success:', 'wootsap'),
                'error_text' => __('Error:', 'wootsap')
            ));
        }
    }

    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        // Add main menu
        add_menu_page(
            __('WooTsap', 'wootsap'),
            __('WooTsap', 'wootsap'),
            'manage_options',
            'wootsap',
            array($this, 'settings_page'),
            'dashicons-whatsapp', // Use WhatsApp icon
            56 // Position after WooCommerce
        );
        
        // Add submenus
        add_submenu_page(
            'wootsap',
            __('WooTsap Settings', 'wootsap'),
            __('Settings', 'wootsap'),
            'manage_options',
            'wootsap-settings',
            array($this, 'settings_page')
        );
        
        add_submenu_page(
            'wootsap',
            __('WooTsap Documentation', 'wootsap'),
            __('Documentation', 'wootsap'),
            'manage_options',
            'wootsap-documentation',
            array($this, 'documentation_page')
        );
        
        add_submenu_page(
            'wootsap',
            __('Buy a Plan', 'wootsap'),
            __('Buy a Plan', 'wootsap'),
            'manage_options',
            'wootsap-buy-plan',
            array($this, 'buy_plan_page')
        );
        
        // Remove duplicate first submenu
        remove_submenu_page('wootsap', 'wootsap');
    }

    /**
     * Register settings
     */
    public function register_settings() {
        // Core settings
        register_setting('wootsap_settings', 'wootsap_api_token');
        register_setting('wootsap_settings', 'wootsap_instance_id');
        register_setting('wootsap_settings', 'wootsap_add_country_code');
        register_setting('wootsap_settings', 'wootsap_default_country_code');
        register_setting('wootsap_settings', 'wootsap_debug_mode');
        
        // Legacy settings (keep for backward compatibility)
        register_setting('wootsap_settings', 'wootsap_enable_processing');
        register_setting('wootsap_settings', 'wootsap_enable_completed');
        register_setting('wootsap_settings', 'wootsap_processing_message');
        register_setting('wootsap_settings', 'wootsap_completed_message');
        
        // Get all WooCommerce order statuses
        if (function_exists('wc_get_order_statuses')) {
            $order_statuses = wc_get_order_statuses();
            
            // Register settings for each status
            foreach ($order_statuses as $status_key => $status_label) {
                $status = str_replace('wc-', '', $status_key);
                register_setting('wootsap_settings', 'wootsap_enable_' . $status);
                register_setting('wootsap_settings', 'wootsap_' . $status . '_message');
            }
        }
    }

    /**
     * Settings page
     */
    public function settings_page() {
        include WOOTSAP_PLUGIN_DIR . 'includes/admin/settings-page.php';
    }
    
    /**
     * Documentation page
     */
    public function documentation_page() {
        include WOOTSAP_PLUGIN_DIR . 'includes/admin/documentation-page.php';
    }
    
    /**
     * Buy plan page
     */
    public function buy_plan_page() {
        include WOOTSAP_PLUGIN_DIR . 'includes/admin/buy-plan-page.php';
    }

    /**
     * Add WhatsApp phone field to checkout
     */
    public function add_whatsapp_phone_field($fields) {
        $fields['billing']['billing_whatsapp'] = array(
            'type'        => 'text',
            'label'       => __('WhatsApp Number', 'wootsap'),
            'placeholder' => __('WhatsApp number for order updates', 'wootsap'),
            'required'    => false,
            'class'       => array('form-row-wide'),
            'clear'       => true,
            'priority'    => 25,
        );
        
        return $fields;
    }

    /**
     * Save WhatsApp phone field - HPOS compatible
     */
    public function save_whatsapp_phone_field($order_id) {
        if (!empty($_POST['billing_whatsapp'])) {
            $order = wc_get_order($order_id);
            if ($order) {
                $order->update_meta_data('_billing_whatsapp', sanitize_text_field($_POST['billing_whatsapp']));
                $order->save();
            }
        }
    }

    /**
     * Add order note checkbox
     */
    public function add_order_note_checkbox() {
        woocommerce_form_field('whatsapp_notification', array(
            'type'      => 'checkbox',
            'class'     => array('input-checkbox'),
            'label'     => __('Send order updates via WhatsApp', 'wootsap'),
            'default'   => '1',
        ), 1);
    }

    /**
     * Save order note checkbox - HPOS compatible
     */
    public function save_order_note_checkbox($order_id) {
        $order = wc_get_order($order_id);
        if ($order) {
            if (isset($_POST['whatsapp_notification'])) {
                $order->update_meta_data('_whatsapp_notification', 'yes');
            } else {
                $order->update_meta_data('_whatsapp_notification', 'no');
            }
            $order->save();
        }
    }

    /**
     * Add meta box to order screen
     */
    public function add_order_meta_box() {
        add_meta_box(
            'wootsap_order_meta_box',
            __('WhatsApp Notifications', 'wootsap'),
            array($this, 'order_meta_box_content'),
            'shop_order',
            'side',
            'default'
        );
    }
    
    /**
     * Order meta box content
     */
    public function order_meta_box_content($post) {
        include WOOTSAP_PLUGIN_DIR . 'includes/admin/order-meta-box.php';
    }
    
    /**
     * Send manual notification via AJAX
     */
    public function send_manual_notification_ajax() {
        // Check nonce
        check_ajax_referer('wootsap-admin', 'security');
        
        // Check user capabilities
        if (!current_user_can('edit_shop_orders')) {
            wp_send_json_error('Permission denied');
            return;
        }
        
        // Get order ID and type
        $order_id = isset($_POST['order_id']) ? intval($_POST['order_id']) : 0;
        $type = isset($_POST['type']) ? sanitize_text_field($_POST['type']) : '';
        
        // Validate order ID
        if (!$order_id) {
            wp_send_json_error('Invalid order ID');
            return;
        }
        
        // Get all possible order statuses
        $valid_types = array();
        if (function_exists('wc_get_order_statuses')) {
            $order_statuses = wc_get_order_statuses();
            foreach ($order_statuses as $status_key => $status_label) {
                $valid_types[] = str_replace('wc-', '', $status_key);
            }
        } else {
            // Fallback to basic statuses
            $valid_types = array('pending', 'processing', 'on-hold', 'completed', 'cancelled', 'refunded', 'failed');
        }
        
        // For backward compatibility, default to processing if type isn't valid
        if (!in_array($type, $valid_types)) {
            $type = 'processing';
        }
        
        // Send notification
        try {
            $result = $this->send_notification($order_id, $type);
            if ($result['success']) {
                wp_send_json_success(sprintf(__('%s notification sent successfully!', 'wootsap'), ucfirst($type)));
            } else {
                wp_send_json_error($result['message']);
            }
        } catch (Exception $e) {
            wp_send_json_error($e->getMessage());
        }
    }
    
    /**
     * Handle order status changes
     */
    public function order_status_changed($order_id, $old_status, $new_status) {
        if (get_option('wootsap_debug_mode') == '1') {
            error_log('WooTsap Debug - Status changed from ' . $old_status . ' to ' . $new_status . ' for order #' . $order_id);
        }
        
        // Check if notifications are enabled for this status
        $enable_option = 'wootsap_enable_' . $new_status;
        
        // For backward compatibility
        if ($new_status == 'processing' && get_option('wootsap_enable_processing') == '1') {
            // Processing is enabled via legacy option
        } else if ($new_status == 'completed' && get_option('wootsap_enable_completed') == '1') {
            // Completed is enabled via legacy option
        } else if (get_option($enable_option) != '1') {
            // This status is not enabled
            if (get_option('wootsap_debug_mode') == '1') {
                error_log('WooTsap Debug - Notifications disabled for status: ' . $new_status);
            }
            return;
        }
        
        // Get the appropriate message option
        $message_option = 'wootsap_' . $new_status . '_message';
        
        // Send notification
        $result = $this->send_notification($order_id, $new_status, $message_option);
        
        if (get_option('wootsap_debug_mode') == '1') {
            error_log('WooTsap Debug - Notification result for ' . $new_status . ': ' . ($result['success'] ? 'Success' : 'Failed - ' . $result['message']));
        }
        
        return $result;
    }
    
    /**
     * Test API connection through AJAX
     */
    public function test_api_ajax_handler() {
        // Check nonce
        check_ajax_referer('wootsap-admin', 'security');
        
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
            return;
        }
        
        // Get API settings
        $token = get_option('wootsap_api_token');
        $instance_id = get_option('wootsap_instance_id');
        
        // Check if settings are valid
        if (empty($token) || empty($instance_id)) {
            wp_send_json_error('API token or instance ID not configured');
            return;
        }
        
        // Get test phone number from the form
        $test_phone = isset($_POST['test_phone']) ? sanitize_text_field($_POST['test_phone']) : '';
        
        // Validate phone number
        if (empty($test_phone)) {
            wp_send_json_error('Please enter a valid phone number for testing');
            return;
        }
        
        // Format the phone number
        $test_phone = $this->format_phone_number($test_phone);
        
        // If number is not valid, return error
        if (!$test_phone) {
            wp_send_json_error('Invalid phone number format. Please check the number and try again.');
            return;
        }
        
        $jid = $test_phone . '@s.whatsapp.net';
        $test_message = "This is a test message from your WooCommerce store. " . date('Y-m-d H:i:s');
        
        // Prepare URL for sending a test message
        $test_url = 'https://api.wootsap.com/api/v1/send-text?token=' . urlencode($token) . 
                    '&instance_id=' . urlencode($instance_id) . 
                    '&jid=' . urlencode($jid) . 
                    '&msg=' . urlencode($test_message);
        
        // Debug log
        if (get_option('wootsap_debug_mode') == '1') {
            error_log('WooTsap Debug - Testing API connection with URL: ' . preg_replace('/token=.*?&/', 'token=HIDDEN&', $test_url));
        }
        
        // Send request
        $response = wp_remote_get($test_url, array(
            'timeout' => 30,
        ));
        
        // Check for errors
        if (is_wp_error($response)) {
            if (get_option('wootsap_debug_mode') == '1') {
                error_log('WooTsap Debug - API Test Error: ' . $response->get_error_message());
            }
            wp_send_json_error('Connection error: ' . $response->get_error_message());
            return;
        }
        
        // Get response body
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        // Debug log
        if (get_option('wootsap_debug_mode') == '1') {
            error_log('WooTsap Debug - API Test Response: ' . $body);
        }
        
        // Check response
        if (!$data || !isset($data['success'])) {
            wp_send_json_error('Invalid API response. Please check your API token and instance ID.');
            return;
        }
        
        if ($data['success']) {
            wp_send_json_success('Test message sent successfully!');
        } else {
            $error_message = isset($data['message']) ? $data['message'] : 'Unknown error';
            wp_send_json_error('API error: ' . $error_message);
        }
    }

    /**
     * For backward compatibility - Processing notification
     */
    public function send_processing_notification($order_id) {
        if (get_option('wootsap_debug_mode') == '1') {
            error_log('WooTsap Debug - Legacy processing hook triggered for order #' . $order_id);
        }
        
        // Check if processing notifications are enabled
        if (get_option('wootsap_enable_processing') != '1') {
            if (get_option('wootsap_debug_mode') == '1') {
                error_log('WooTsap Debug - Processing notifications disabled in settings');
            }
            return array('success' => false, 'message' => 'Processing notifications disabled');
        }

        return $this->send_notification($order_id, 'processing');
    }

    /**
     * For backward compatibility - Completed notification
     */
    public function send_completed_notification($order_id) {
        if (get_option('wootsap_debug_mode') == '1') {
            error_log('WooTsap Debug - Legacy completed hook triggered for order #' . $order_id);
        }
        
        // Check if completed notifications are enabled
        if (get_option('wootsap_enable_completed') != '1') {
            if (get_option('wootsap_debug_mode') == '1') {
                error_log('WooTsap Debug - Completed notifications disabled in settings');
            }
            return array('success' => false, 'message' => 'Completed notifications disabled');
        }

        return $this->send_notification($order_id, 'completed');
    }

    /**
     * Send notification - HPOS compatible
     */
    public function send_notification($order_id, $status, $message_option = '') {
        if (get_option('wootsap_debug_mode') == '1') {
            error_log('WooTsap Debug - Sending ' . $status . ' notification for order #' . $order_id);
        }
        
        // Get order
        $order = wc_get_order($order_id);
        if (!$order) {
            if (get_option('wootsap_debug_mode') == '1') {
                error_log('WooTsap Debug - Error: Order #' . $order_id . ' not found');
            }
            return array('success' => false, 'message' => 'Order not found');
        }
        
        // Check if WhatsApp notification is enabled for this order - HPOS compatible
        $whatsapp_notification = $order->get_meta('_whatsapp_notification', true);
        if (get_option('wootsap_debug_mode') == '1') {
            error_log('WooTsap Debug - WhatsApp notification meta: ' . $whatsapp_notification);
        }
        
        if ($whatsapp_notification == 'no') {
            if (get_option('wootsap_debug_mode') == '1') {
                error_log('WooTsap Debug - Notification disabled by customer');
            }
            // Add note to order
            $order->add_order_note(__('WhatsApp notification skipped (disabled by customer).', 'wootsap'));
            return array('success' => false, 'message' => 'Notification disabled by customer');
        }
        
        // Get WhatsApp number - HPOS compatible
        $whatsapp_number = $order->get_meta('_billing_whatsapp', true);
        if (get_option('wootsap_debug_mode') == '1') {
            error_log('WooTsap Debug - WhatsApp number from meta: ' . $whatsapp_number);
        }
        
        // If no WhatsApp number provided, use billing phone
        if (empty($whatsapp_number)) {
            $whatsapp_number = $order->get_billing_phone();
            if (get_option('wootsap_debug_mode') == '1') {
                error_log('WooTsap Debug - Using billing phone instead: ' . $whatsapp_number);
            }
        }
        
        // If still no phone number, add note and return
        if (empty($whatsapp_number)) {
            if (get_option('wootsap_debug_mode') == '1') {
                error_log('WooTsap Debug - No phone number provided');
            }
            $order->add_order_note(__('WhatsApp notification skipped (no phone number).', 'wootsap'));
            return array('success' => false, 'message' => 'No phone number provided');
        }

        // Format WhatsApp number
        $original_number = $whatsapp_number;
        $whatsapp_number = $this->format_phone_number($whatsapp_number);
        if (get_option('wootsap_debug_mode') == '1') {
            error_log('WooTsap Debug - Phone number formatted: ' . $original_number . ' -> ' . $whatsapp_number);
        }
        
        // If number is not valid, add note and return
        if (!$whatsapp_number) {
            if (get_option('wootsap_debug_mode') == '1') {
                error_log('WooTsap Debug - Invalid phone number');
            }
            $order->add_order_note(__('WhatsApp notification skipped (invalid phone number).', 'wootsap'));
            return array('success' => false, 'message' => 'Invalid phone number');
        }
        
        // Get appropriate message template
        $message_template = '';
        
        // If a specific message option key was provided
        if (!empty($message_option)) {
            $message_template = get_option($message_option, '');
            if (get_option('wootsap_debug_mode') == '1') {
                error_log('WooTsap Debug - Using specific message template from option: ' . $message_option);
            }
        }
        
        // If no specific template or it's empty, fall back to legacy options or defaults
        if (empty($message_template)) {
            if ($status == 'processing') {
                $message_template = get_option('wootsap_processing_message');
                if (get_option('wootsap_debug_mode') == '1') {
                    error_log('WooTsap Debug - Using legacy processing message template');
                }
            } else if ($status == 'completed') {
                $message_template = get_option('wootsap_completed_message');
                if (get_option('wootsap_debug_mode') == '1') {
                    error_log('WooTsap Debug - Using legacy completed message template');
                }
            } else {
                // For other statuses, create a default message
                $status_label = wc_get_order_status_name($status);
                $message_template = 'Hello {customer_name}, the status of your order #{order_number} has been updated to ' . $status_label . '. Total: {order_total}.';
                if (get_option('wootsap_debug_mode') == '1') {
                    error_log('WooTsap Debug - Using default message template for status: ' . $status);
                }
            }
        }
        
        if (get_option('wootsap_debug_mode') == '1') {
            error_log('WooTsap Debug - Message template before variable replacement: ' . $message_template);
        }
        
        // Replace
        $message = $this->replace_variables($message_template, $order);
        if (get_option('wootsap_debug_mode') == '1') {
            error_log('WooTsap Debug - Final message after variable replacement: ' . $message);
        }
        
        // Send WhatsApp message
        $result = $this->send_whatsapp_message($whatsapp_number, $message);
        
        // Add note to order
        if (isset($result['success']) && $result['success']) {
            if (get_option('wootsap_debug_mode') == '1') {
                error_log('WooTsap Debug - Message sent successfully');
            }
            $status_label = wc_get_order_status_name($status);
            $order->add_order_note(sprintf(__('WhatsApp notification for %s status sent to %s.', 'wootsap'), $status_label, $whatsapp_number));
        } else {
            $error_message = isset($result['message']) ? $result['message'] : 'Unknown error';
            if (get_option('wootsap_debug_mode') == '1') {
                error_log('WooTsap Debug - Failed to send message: ' . $error_message);
            }
            $status_label = wc_get_order_status_name($status);
            $order->add_order_note(sprintf(__('Failed to send WhatsApp notification for %s status: %s', 'wootsap'), $status_label, $error_message));
        }
        
        return $result;
    }

    /**
     * Format phone number
     */
    public function format_phone_number($phone) {
        // Remove any non-numeric characters
        $phone = preg_replace('/[^0-9]/', '', $phone);
        
        // Check if number is valid
        if (strlen($phone) < 7) {
            return false;
        }
        
        // Add country code if enabled and not already present
        if (get_option('wootsap_add_country_code') == '1') {
            $default_country_code = get_option('wootsap_default_country_code', '91');
            
            // Check if country code is already added
            if (substr($phone, 0, strlen($default_country_code)) !== $default_country_code) {
                $phone = $default_country_code . $phone;
            }
        }
        
        return $phone;
    }

    /**
     * Replace variables in message - Fixed version to handle HTML properly
     */
    public function replace_variables($message, $order) {
        // Get raw data without HTML formatting
        $customer_name = $order->get_billing_first_name();
        $order_number = $order->get_order_number();
        $order_status = wc_get_order_status_name($order->get_status());
        $site_name = get_bloginfo('name');
        
        // Get currency symbol and amount separately to avoid HTML formatting
        $order_total = $order->get_total();
        $currency_symbol = get_woocommerce_currency_symbol();
        
        // Format total without HTML
        $formatted_total = $currency_symbol . number_format($order_total, 2);
        
        // If there are any HTML entities in the currency symbol (like &pound;), decode them
        $formatted_total = html_entity_decode($formatted_total);
        
        // Create replacements array
        $replacements = array(
            '{customer_name}' => $customer_name,
            '{order_number}' => $order_number,
            '{order_total}' => $formatted_total,
            '{order_status}' => $order_status,
            '{site_name}' => $site_name,
        );
        
        // Replace all variables
        $final_message = str_replace(array_keys($replacements), array_values($replacements), $message);
        
        // Strip any remaining HTML
        $final_message = strip_tags($final_message);
        
        // Convert HTML entities to their actual characters
        $final_message = html_entity_decode($final_message);
        
        return $final_message;
    }

    /**
     * Send WhatsApp message
     */
    public function send_whatsapp_message($phone, $message) {
        // Get API settings
        $token = get_option('wootsap_api_token');
        $instance_id = get_option('wootsap_instance_id');
        
        // Debug log
        if (get_option('wootsap_debug_mode') == '1') {
            error_log('WooTsap Debug - Attempting to send message');
            error_log('WooTsap Debug - Phone: ' . $phone);
            error_log('WooTsap Debug - Token: ' . substr($token, 0, 5) . '...');
            error_log('WooTsap Debug - Instance ID: ' . $instance_id);
        }
        
        // Check if settings are valid
        if (empty($token) || empty($instance_id)) {
            if (get_option('wootsap_debug_mode') == '1') {
                error_log('WooTsap Debug - Error: API token or instance ID not configured');
            }
            return array(
                'success' => false,
                'message' => __('API token or instance ID not configured.', 'wootsap')
            );
        }
        
        // Format JID
        $jid = $phone . '@s.whatsapp.net';
        
        // Prepare URL - Use direct concatenation to avoid issues
        $url = 'https://api.wootsap.com/api/v1/send-text?token=' . urlencode($token) . 
               '&instance_id=' . urlencode($instance_id) . 
               '&jid=' . urlencode($jid) . 
               '&msg=' . urlencode($message);
        
        if (get_option('wootsap_debug_mode') == '1') {
            error_log('WooTsap Debug - Request URL: ' . preg_replace('/token=.*?&/', 'token=HIDDEN&', $url));
        }
        
        // Send request
        $response = wp_remote_get($url, array(
            'timeout' => 30,
        ));
        
        // Check for errors
        if (is_wp_error($response)) {
            if (get_option('wootsap_debug_mode') == '1') {
                error_log('WooTsap Debug - WP Error: ' . $response->get_error_message());
            }
            return array(
                'success' => false,
                'message' => $response->get_error_message()
            );
        }
        
        // Get response body
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        if (get_option('wootsap_debug_mode') == '1') {
            error_log('WooTsap Debug - Response Code: ' . wp_remote_retrieve_response_code($response));
            error_log('WooTsap Debug - Response Body: ' . $body);
        }
        
        // Check response
        if (!$data || !isset($data['success'])) {
            if (get_option('wootsap_debug_mode') == '1') {
                error_log('WooTsap Debug - Invalid API response');
            }
            return array(
                'success' => false,
                'message' => __('Invalid API response.', 'wootsap')
            );
        }
        
        if (get_option('wootsap_debug_mode') == '1') {
            error_log('WooTsap Debug - API Response Success: ' . ($data['success'] ? 'true' : 'false'));
            if (isset($data['message'])) {
                error_log('WooTsap Debug - API Message: ' . $data['message']);
            }
        }
        
        return $data;
    }
}